# 機能設計書 3-線形代数演算（Linear Algebra Operations）

## 概要

本ドキュメントは、TensorFlowのコアフレームワークにおける線形代数演算（Linear Algebra Operations）機能の設計を記述する。行列分解・逆行列・固有値分解・特異値分解などの線形代数演算を提供する。

### 本機能の処理概要

線形代数演算機能は、行列やテンソルに対する高度な線形代数的演算を提供する。行列式計算、逆行列、コレスキー分解、LU分解、QR分解、特異値分解（SVD）、固有値分解、行列方程式の求解など、科学計算や機械学習で頻繁に使用される線形代数操作を網羅する。

**業務上の目的・背景**：主成分分析（PCA）、正規方程式によるパラメータ推定、共分散行列の分解、カルマンフィルタなど、多くの機械学習・信号処理アルゴリズムは線形代数演算を基盤とする。本機能はこれらをバッチ行列に対して効率的に実行する手段を提供し、大規模なデータセットに対する線形代数操作を可能にする。

**機能の利用シーン**：(1) PCAのための特異値分解、(2) ガウス過程の共分散行列コレスキー分解、(3) 線形回帰の正規方程式求解、(4) アインシュタインの縮約記法による汎用テンソル演算、(5) 行列式による正則性判定。

**主要な処理内容**：
1. 行列式計算: MatrixDeterminant, LogMatrixDeterminant
2. 逆行列: MatrixInverse
3. コレスキー分解: Cholesky, CholeskyGrad
4. 固有値分解: Eig, SelfAdjointEigV2
5. LU分解: Lu
6. QR分解: Qr
7. 特異値分解: Svd
8. 行列方程式求解: MatrixSolve, MatrixTriangularSolve, BandedTriangularSolve, MatrixSolveLs
9. 行列指数・対数: MatrixExponential, MatrixLogarithm
10. アインシュタイン縮約: Einsum
11. 三重対角行列演算: TridiagonalMatMul, TridiagonalSolve

**関連システム・外部連携**：LAPACK互換のC++カーネルを通じて実行される。GPU上ではcuSOLVER/cuBLASライブラリが利用される。すべての操作はバッチ次元をサポートし、複数の行列に対する並列処理が可能である。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はコアAPIのため直接的な画面関連はなし |

## 機能種別

計算処理（線形代数演算処理）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input | tf.Tensor | Yes | 入力行列（2次元以上のテンソル） | ランク2以上であること |
| adjoint | bool | No | 転置共役を使用するか | - |
| full_matrices | bool | No | 完全分解か経済分解か | - |
| equation | string | Einsum時 | アインシュタイン縮約式 | 有効な縮約式であること |

### 入力データソース

Python API呼び出し時の引数として入力される。入力は原則として浮動小数点型（float32, float64, complex64, complex128）の行列テンソルである。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | tf.Tensor | 分解結果、求解結果のテンソル |
| sign / log_det | tf.Tensor | LogMatrixDeterminant時の符号と対数行列式 |
| eigenvalues | tf.Tensor | 固有値（Eig/SelfAdjointEigV2時） |
| eigenvectors | tf.Tensor | 固有ベクトル（Eig/SelfAdjointEigV2時） |

### 出力先

呼び出し元のPythonコードに返却される。

## 処理フロー

### 処理シーケンス

```
1. Python API呼び出し（tf.linalg名前空間）
   └─ 入力テンソルの型・形状検証
2. バッチ次元の検出
   └─ 入力テンソルの末尾2次元を行列として扱い、先頭次元をバッチ
3. 正方行列検証（必要な操作のみ）
   └─ MakeBatchSquareMatrix()による形状チェック
4. C++カーネル呼び出し
   └─ CPU: LAPACK / GPU: cuSOLVER
5. 結果テンソル返却
```

### フローチャート

```mermaid
flowchart TD
    A[tf.linalg API呼び出し] --> B[入力テンソル検証]
    B --> C{正方行列必要?}
    C -->|Yes| D[正方行列チェック]
    C -->|No| E[一般行列チェック]
    D --> F[バッチ次元検出]
    E --> F
    F --> G{GPU利用可能?}
    G -->|Yes| H[cuSOLVER実行]
    G -->|No| I[LAPACK実行]
    H --> J[結果返却]
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 正方行列制約 | 行列式・逆行列・コレスキー分解等は正方行列を要求 | 対象操作時 |
| BR-02 | 正定値制約 | コレスキー分解は正定値対称行列を要求 | Cholesky操作時 |
| BR-03 | バッチサポート | すべての操作がバッチ次元（先頭次元）をサポート | 常時 |
| BR-04 | バンド幅制約 | BandedTriangularSolveではバンド数が行列サイズ以下 | BandedTriangularSolve時 |

### 計算ロジック

- Cholesky: A = L * L^T（Lは下三角行列）
- SVD: A = U * S * V^T
- QR: A = Q * R（Qは直交行列、Rは上三角行列）
- Einsum: アインシュタインの縮約記法に基づく汎用テンソル演算

## データベース操作仕様

### 操作別データベース影響一覧

| 操作 | 対象テーブル | 操作種別 | 概要 |
|-----|-------------|---------|------|
| - | - | - | データベース操作は発生しない |

### テーブル別操作詳細

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgumentError | 形状エラー | 非正方行列を正方行列専用操作に入力 | 正方行列を入力する |
| InvalidArgumentError | 数値エラー | 特異行列の逆行列を計算しようとした | 正則行列を使用する、または擬似逆行列を使用 |
| InvalidArgumentError | バンド幅エラー | BandedTriangularSolveでバンド数が行列サイズを超過 | バンド数を行列サイズ以下にする |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- バッチ次元に対する並列処理により、複数行列の同時分解を効率化すること
- GPU上ではcuSOLVERを活用し、CPU上ではLAPACKを活用すること

## セキュリティ考慮事項

- 大規模行列の分解によるメモリ枯渇への対策
- 数値的不安定性（ill-conditioned matrix）への対処

## 備考

- Batch接頭辞を持つopは後方互換性のために残されている（BatchCholesky等）
- Einsum操作は非常に汎用的で、行列積やトレース、テンソル縮約など多くの操作を一つの式で表現可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

線形代数操作が扱うバッチ行列の形状規約を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | linalg_ops.cc | `tensorflow/core/ops/linalg_ops.cc` | MakeBatchSquareMatrix関数による形状検証 |

**読解のコツ**: 線形代数操作は末尾2次元を行列として扱い、それ以前の次元をバッチ次元として並列処理する。`[..., M, N]` 形式の入力テンソルが基本形である。

#### Step 2: エントリーポイントを理解する

C++のop定義から操作の入出力仕様を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | linalg_ops.cc | `tensorflow/core/ops/linalg_ops.cc` | REGISTER_OPによるop定義一覧 |

**主要処理フロー**:
- **29-41行目**: `MakeBatchSquareMatrix` - バッチ正方行列の形状検証ユーティリティ
- **43-48行目**: `BatchUnchangedSquareShapeFn` - 正方行列不変形状推論
- **51-91行目**: `BandedTriangularSolveShapeFn` - バンド三角求解の形状推論
- **95行目以降**: `MatrixSolveShapeFn` - 行列求解の形状推論
- **362行目**: `MatrixDeterminant` op登録
- **380行目**: `LogMatrixDeterminant` op登録
- **403行目**: `MatrixInverse` op登録
- **424行目**: `Cholesky` op登録
- **457行目**: `Eig` op登録
- **474行目**: `Lu` op登録
- **482行目**: `MatrixSolve` op登録
- **533行目**: `Qr` op登録
- **541行目**: `Svd` op登録
- **569行目**: `Einsum` op登録

#### Step 3: バッチ対応の仕組みを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | linalg_ops.cc | `tensorflow/core/ops/linalg_ops.cc` | Batch接頭辞を持つレガシーop群（580行目以降） |

**読解のコツ**: `BatchCholesky`, `BatchMatrixSolve` 等のBatch接頭辞opは後方互換性のために残されている。新しいコードでは接頭辞なしのop（Cholesky, MatrixSolve等）がバッチ次元を自動的にサポートする。

### プログラム呼び出し階層図

```
tf.linalg.cholesky(input)
    |
    +-- gen_linalg_ops.cholesky(input)
            |
            +-- C++ Cholesky kernel
                    |
                    +-- BatchUnchangedSquareShapeFn (形状推論)
                    +-- CPU: LAPACK dpotrf
                    +-- GPU: cuSOLVER cusolverDnDpotrf
```

### データフロー図

```
[入力]                    [処理]                        [出力]

バッチ行列         -->  linalg_ops.cc              -->  分解結果
[..., M, N]            (形状推論・検証)                 [..,..]
                            |
                    C++ Kernel
                    (LAPACK/cuSOLVER)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| linalg_ops.cc | `tensorflow/core/ops/linalg_ops.cc` | ソース | C++オペレーション定義・形状推論 |
| gen_linalg_ops.py | `tensorflow/python/ops/gen_linalg_ops.py` | 自動生成 | Python バインディング |
| linalg_impl.py | `tensorflow/python/ops/linalg/linalg_impl.py` | ソース | Python高レベルAPI |
| common_shape_fns.h | `tensorflow/core/framework/common_shape_fns.h` | ヘッダ | 共通形状推論関数 |
